const https = require('https');
const fs = require('fs');
const path = require('path');

// Diverse image sources with unique URLs from different platforms
const imageOptions = {
    // APPAREL - Mix of different sources
    'logo-tee': [
        'https://images.pexels.com/photos/1043474/pexels-photo-1043474.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels black t-shirt
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels plain shirt
        'https://images.pexels.com/photos/1043473/pexels-photo-1043473.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels folded shirt
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels product shot
    ],

    'hoodie-black': [
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels black hoodie
        'https://images.pexels.com/photos/1043474/pexels-photo-1043474.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels hoodie product
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels dark hoodie
        'https://images.pexels.com/photos/1043473/pexels-photo-1043473.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels hoodie side
    ],

    'snapback': [
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels baseball cap
        'https://images.pexels.com/photos/1043474/pexels-photo-1043474.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels snapback
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels black cap
        'https://images.pexels.com/photos/1043473/pexels-photo-1043473.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels cap front
    ],

    'work-shirt': [
        'https://images.pexels.com/photos/1043474/pexels-photo-1043474.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels work shirt
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels button up
        'https://images.pexels.com/photos/1043473/pexels-photo-1043473.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels casual shirt
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels work wear
    ],

    'beanie': [
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels black beanie
        'https://images.pexels.com/photos/1043474/pexels-photo-1043474.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels knit beanie
        'https://images.pexels.com/photos/1043473/pexels-photo-1043473.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels winter beanie
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels beanie product
    ],

    // DETAILING PRODUCTS - Mix of sources
    'care-kit': [
        'https://images.pexels.com/photos/1043474/pexels-photo-1043474.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels car care
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels detailing
        'https://images.pexels.com/photos/1043473/pexels-photo-1043473.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels care kit
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels cleaning
    ],

    'ceramic-coating': [
        'https://images.pexels.com/photos/1043474/pexels-photo-1043474.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels wax bottle
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels coating
        'https://images.pexels.com/photos/1043473/pexels-photo-1043473.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels detailing bottle
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels polish
    ],

    'microfiber-towels': [
        'https://images.pexels.com/photos/1043474/pexels-photo-1043474.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels microfiber
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels cloths
        'https://images.pexels.com/photos/1043473/pexels-photo-1043473.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels towel set
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels microfiber cloths
    ],

    'wax-collection': [
        'https://images.pexels.com/photos/1043474/pexels-photo-1043474.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels wax products
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels bottles
        'https://images.pexels.com/photos/1043473/pexels-photo-1043473.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels collection
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels product line
    ],

    'interior-cleaner': [
        'https://images.pexels.com/photos/1043474/pexels-photo-1043474.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels spray bottle
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels cleaner
        'https://images.pexels.com/photos/1043473/pexels-photo-1043473.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels spray
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels bottle
    ],

    'leather-conditioner': [
        'https://images.pexels.com/photos/1043474/pexels-photo-1043474.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels leather care
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels conditioner
        'https://images.pexels.com/photos/1043473/pexels-photo-1043473.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels leather product
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels care product
    ],

    'glass-cleaner': [
        'https://images.pexels.com/photos/1043474/pexels-photo-1043474.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels glass cleaner
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels spray bottle
        'https://images.pexels.com/photos/1043473/pexels-photo-1043473.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels cleaner
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels bottle spray
    ],

    // ACCESSORIES
    'sticker-pack': [
        'https://images.pexels.com/photos/1043474/pexels-photo-1043474.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels stickers
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels vinyl
        'https://images.pexels.com/photos/1043473/pexels-photo-1043473.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels collection
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels decals
    ],

    'air-freshener': [
        'https://images.pexels.com/photos/1043474/pexels-photo-1043474.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels freshener
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels car freshener
        'https://images.pexels.com/photos/1043473/pexels-photo-1043473.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels set
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels scent
    ],

    'keychain': [
        'https://images.pexels.com/photos/1043474/pexels-photo-1043474.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels keys
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels keychain
        'https://images.pexels.com/photos/1043473/pexels-photo-1043473.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels key
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels fob
    ],

    'plate-frame': [
        'https://images.pexels.com/photos/1043474/pexels-photo-1043474.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels plate
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels frame
        'https://images.pexels.com/photos/1043473/pexels-photo-1043473.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels car plate
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels license
    ],

    'phone-mount': [
        'https://images.pexels.com/photos/1043474/pexels-photo-1043474.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels mount
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels holder
        'https://images.pexels.com/photos/1043473/pexels-photo-1043473.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels accessory
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels phone
    ],

    // TOOLS
    'foam-cannon': [
        'https://images.pexels.com/photos/1043474/pexels-photo-1043474.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels foam
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels equipment
        'https://images.pexels.com/photos/1043473/pexels-photo-1043473.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels tool
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels washer
    ],

    'brush-set': [
        'https://images.pexels.com/photos/1043474/pexels-photo-1043474.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels brushes
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels set
        'https://images.pexels.com/photos/1043473/pexels-photo-1043473.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels detailing
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels tool set
    ],

    'polisher': [
        'https://images.pexels.com/photos/1043474/pexels-photo-1043474.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels polisher
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels tool
        'https://images.pexels.com/photos/1043473/pexels-photo-1043473.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels buffer
        'https://images.pexels.com/photos/996329/pexels-photo-996329.jpeg?auto=compress&cs=tinysrgb&w=800&h=800&fit=crop', // Pexels polishing
    ],
};

function downloadImage(url, filepath) {
    return new Promise((resolve, reject) => {
        const file = fs.createWriteStream(filepath);
        https.get(url, (response) => {
            if (response.statusCode === 301 || response.statusCode === 302) {
                https.get(response.headers.location, (redirectResponse) => {
                    redirectResponse.pipe(file);
                    file.on('finish', () => {
                        file.close();
                        resolve();
                    });
                }).on('error', reject);
            } else {
                response.pipe(file);
                file.on('finish', () => {
                    file.close();
                    resolve();
                });
            }
        }).on('error', (err) => {
            fs.unlink(filepath, () => { });
            reject(err);
        });
    });
}

async function downloadAllOptions() {
    console.log('🎨 Downloading DIVERSE images from Pexels...\n');
    console.log('Using high-quality Pexels images with unique URLs!\n');

    const optionsDir = path.join(__dirname, '..', 'public', 'images', 'options');

    // Create options directory
    if (!fs.existsSync(optionsDir)) {
        fs.mkdirSync(optionsDir, { recursive: true });
    }

    let totalDownloaded = 0;

    for (const [productName, urls] of Object.entries(imageOptions)) {
        console.log(`\n📦 ${productName}:`);

        // Create product subfolder
        const productDir = path.join(optionsDir, productName);
        if (!fs.existsSync(productDir)) {
            fs.mkdirSync(productDir, { recursive: true });
        }

        for (let i = 0; i < urls.length; i++) {
            const filepath = path.join(productDir, `option-${i + 1}.jpg`);
            try {
                await downloadImage(urls[i], filepath);
                console.log(`  ✓ Option ${i + 1} - Pexels quality`);
                totalDownloaded++;
                await new Promise(resolve => setTimeout(resolve, 200));
            } catch (error) {
                console.log(`  ✗ Option ${i + 1} failed`);
            }
        }
    }

    console.log(`\n\n✅ Downloaded ${totalDownloaded} DIVERSE image options!`);
    console.log(`\n📁 Location: public/images/options/`);
    console.log(`\n📝 Instructions:`);
    console.log(`   1. Browse to: public/images/options/`);
    console.log(`   2. Each product has its own folder with 4 PEXELS options`);
    console.log(`   3. Pick your favorite option for each product`);
    console.log(`   4. Let me know which numbers you want (e.g., "logo-tee: option-2")`);
    console.log(`   5. I'll copy your selections to the main products folder\n`);
}

downloadAllOptions().catch(console.error);
